using System;
using System.Collections.Generic;
using System.Text;
using ESRI.ArcGIS.AnalysisTools;
using ESRI.ArcGIS.DataManagementTools;
using ESRI.ArcGIS.Geoprocessor;
using ESRI.ArcGIS.Geoprocessing;
using ESRI.ArcGIS.esriSystem;
using ESRI.ArcGIS.Geodatabase;
using ESRI.ArcGIS.DataSourcesFile;
using ESRI.ArcGIS.DataSourcesGDB;
using System.IO;

namespace GpExampleToolboxTools
{
    class Program
    {
        static void Main(string[] args)
        {
            Geoprocessor gp = new Geoprocessor();
            // Set the workspace
            gp.SetEnvironmentValue("workspace", @"C:\Data\ArkansasData.gdb");
            // Set up StreamWriter to log to
            StreamWriter lw = File.AppendText(@"C:\Data\Logfile_" + 
                              DateTime.Now.ToString("yyyy_MM_dd_hh_mm_ss_tt") + ".log");
            // Setup array with featureclasses we need to convert to layers
            string[] fcs = new string[2] { "NhdMediumResLinesBeaver", 
                                           "NhdMediumResLinesIllinois" };
            // Iterate through featureclasses, process each one
            foreach (string fc in fcs)
            {
                // Make feature layers for input into SelectByAttribute tool
                MakeFeatureLayer mfl = new MakeFeatureLayer();
                // Set input and output parameters
                mfl.in_features = fc;
                mfl.out_layer = fc + "_Lyr";
                // Run the tool
                RunTool(gp, mfl, null);

                // Run SelectByAttribute to find perennial streams
                SelectLayerByAttribute sba = new SelectLayerByAttribute();
                // Set input and output parameters
                sba.in_layer_or_view = fc + "_Lyr";
                sba.selection_type = "NEW_SELECTION";
                // Note that in where_clause we must escape double quotes around
                //  field name with whacks (backward slashes)
                sba.where_clause = "\"FCODE_DESC\" = 'Stream/River: " + 
                                   "Hydrographic Category = Perennial'";
                RunTool(gp, sba, null);

                // Run SelectLayerByLocation tool to find perennial stream 
                //  segments that intersect city limits
                SelectLayerByLocation sbl = new SelectLayerByLocation();
                sbl.in_layer = fc + "_Lyr";
                // Do a selection on the subset that we already selected above 
                //  by attribute (select on the
                //  already existing selection set)
                sbl.selection_type = "SUBSET_SELECTION";
                sbl.select_features = "ArkansasCityLimits";
                sbl.overlap_type = "INTERSECT";
                RunTool(gp, sbl, null);

                // Run Clip tool - note that we must fully qualify the 
                //  toolbox reference as there is also a
                //  Clip tool in the Data Management toolbox
                ESRI.ArcGIS.AnalysisTools.Clip clip = 
                                   new ESRI.ArcGIS.AnalysisTools.Clip();
                clip.in_features = fc + "_Lyr";
                // Clip the the city limits layer
                clip.clip_features = "ArkansasCityLimits";
                // Push out a new featureclass to our file geodatabase
                clip.out_feature_class = fc + "_WithinCityLimits";
                RunTool(gp, clip, null);

                // Get a sum of the stream lengths
                double sum = SumStreamLengths(fc + "_WithinCityLimits");
                string msg = "Stream length sum for featureclass \"" + fc + 
                             "_WithinCityLimits\": " + sum.ToString() + " KM";
                Console.WriteLine(msg);
                Log(msg, lw);
            }          
        }

        // Method to iterate through a featureclass and sum the stream lengths
        private static double SumStreamLengths(string inputFc)
        {
            // Get a featureclass object
            IFeatureClass fc = OpenFc(@"C:\Data\ArkansasData.gdb", inputFc);
            // Set up a queryfilter on the LENGTHKM field
            IQueryFilter2 qf = new QueryFilterClass();
            qf.SubFields = "LENGTHKM";
            int fieldPositionLenghtKm = fc.FindField("LENGTHKM");
            // Create a featurecursor to loop through
            IFeatureCursor featCur = fc.Search(qf, false);
            IDataset ds = (IDataset)fc;
            IFeature feature = null;
            double k = 0;
            // Start looping through the features
            while ((feature = featCur.NextFeature()) != null)
            {
                // Get the value of the LENGTHKM field for each feature
                double lenKm = Convert.ToDouble(
                              feature.get_Value(fieldPositionLenghtKm));
                // Sum the values with every iteration
                k = k + lenKm;            
            }
            // Return the final summation of the stream lengths
            return k;
        }

        // Method to open up a featureclass and return it as an object
        private static IFeatureClass OpenFc(string inputWs, string inputFc)
        {
            // Set up a workspace factory for a file geodatabase
            IWorkspaceFactory2 wsf2 = (IWorkspaceFactory2)new 
                  ESRI.ArcGIS.DataSourcesGDB.FileGDBWorkspaceFactoryClass();
            // Create a workspace in our file geodatabase and open it
            IWorkspace ws = wsf2.OpenFromFile(inputWs, 0);
            IFeatureWorkspace fws = (IFeatureWorkspace)ws;
            // Open up the featureclass, stuff into a IFeatureClass object
            IFeatureClass fc = fws.OpenFeatureClass(inputFc);
            // Return the featureclass object
            return fc;
        }

        // Method to run our geoprocessing tools and handle 
        //  any errors that occur
        private static void RunTool(Geoprocessor geoprocessor, 
                                    IGPProcess process, ITrackCancel TC)
        {
            // Set the overwrite output option to true; avoid issues 
            //  with output datasets that may already exist
            geoprocessor.OverwriteOutput = true;
            // Execute the tool            
            try
            {
                geoprocessor.Execute(process, null);
                // Print out processing messages to stdout
                ReturnMessages(geoprocessor);
            }
            catch (Exception err)
            {
                Console.WriteLine(err.Message);
                ReturnMessages(geoprocessor);
            }
        }

        // Method for returning the tool messages.
        private static void ReturnMessages(Geoprocessor gp)
        {
            if (gp.MessageCount > 0)
            {
                for (int Count = 0; Count <= gp.MessageCount - 1; Count++)
                {
                    Console.WriteLine(gp.GetMessage(Count));
                }
            }
        }

        private static void Log(String logMessage, TextWriter w)
        // Write messages to log file
        {
            w.WriteLine(logMessage);
            w.Flush();
        }
    }
}
